
Type TSweepAndPruneCollider Extends IBroadPhaseCollider

	Field _physics:TPhysicsSimulator
	Field _xExtentList:TExtentList
	Field _yExtentList:TExtentList
	Field _xInfoList:TExtentInfoList
	Field _yInfoList:TExtentInfoList
	Field _collisionPairs:TCollisionPairDictionary
	Global fTol:Float = 0.01
	
	Function Create:TSweepAndPruneCollider(physics:TPhysicsSimulator)
		Local sap:TSweepAndPruneCollider = New TSweepAndPruneCollider
		sap._physics = physics
		sap._xExtentList = TExtentList.Create(sap)
		sap._yExtentList = TExtentList.Create(sap)
		sap._xInfoList = TExtentInfoList.Create(sap)
		sap._yInfoList = TExtentInfoList.Create(sap)
		sap._collisionPairs = New TCollisionPairDictionary
		Return sap
	End Function
	
	rem
	Used by the PhysicsSimulator To remove geometry from Sweep And Prune once it
	has been removed.
	End rem
	Method ProcessRemovedGeoms()
		Local xmatch:Int = False
		For Local i:TExtentInfo = EachIn _xInfoList
			If i._geometry.isRemoved Then
				_xInfoList.Remove(i)
				xMatch = True
			End If
		Next
		
		If xmatch Then
			For Local n:TExtent = EachIn _xExtentList
				If n._info._geometry.isRemoved Then
					_xExtentList.Remove(n)
				End If
			Next
		End If
		
		Local ymatch:Int = False
		For Local i:TExtentInfo = EachIn _yInfoList
			If i._geometry.isRemoved Then
				_yInfoList.Remove(i)
				yMatch = True
			End If
		Next
		If yMatch Then
			For Local n:TExtent = EachIn _yExtentList
				If n._info._geometry.isRemoved Then
					_yExtentList.Remove(n)
				End If
			Next
		End If
		
		rem
		We force a non-incremental update because that will ensure that the
		collisionPairs get recreated and that the geometry isn't being held
		by overlaps, etc. Its just easier this way.
		end rem		
		ForceNonIncrementalUpdate()
		
	End Method
	
	rem
	This method is used by the PhysicsSimulator to notify Sweep and Prune that 
	new geometry is to be tracked.
	end rem
	Method Add(geom:TGeom)
		Local xExtentInfo:TExtentInfo = TExtentInfo.Create(geom, geom._aabb._min.X, geom._aabb._max.X)
		_xInfoList.Add(xExtentInfo)
		_xExtentList.IncrementalInsertExtent(xExtentInfo)
		
		Local yExtentInfo:TExtentInfo = TExtentInfo.Create(geom, geom._aabb._min.Y, geom._aabb._max.Y)
		_yInfoList.Add(yExtentInfo)
		_yExtentList.IncrementalInsertExtent(yExtentInfo)
	End Method
	
	rem
	Test AABB collisions between two geometries. Tests include checking if the
	geometries are enabled, static, in the right collision categories, etc.
	returns: true if collision, false otherwise
	end rem
	Function DoCollision:Int(g1:TGeom, g2:TGeom)
		If Not(g1._body._enabled) Or Not(g2._body._enabled) Then
			Return False
		End If
		
		If (g1._collisionGroup = g2._collisionGroup) And (g1._collisionGroup <> 0 And g2._collisionGroup <> 0) Then
			Return False
		End If
		
		If Not(g1._collisionEnabled) Or Not(g2._collisionEnabled) Then
			Return False
		End If
		
		If g1._body._isStatic And g2._body._isStatic Then
			Return False
		End If
		
		If g1._body = g2._body Then
			Return False
		End If
		
		If ((g1._collisionCategories & g2._collidesWith) = CollisionCategories.None) & ((g2._collisionCategories & g1._collidesWith) = CollisionCategories.None) Then
			Return False
		End If
		
		'temp vars
		Local aabb1:TAABB = New TAABB
		Local aabb2:TAABB = New TAABB
		aabb1._Min.Set(g1._aabb._Min)
		aabb1._Max.Set(g1._aabb._Max)
		aabb2._Min.Set(g2._aabb._Min)
		aabb2._Max.Set(g2._aabb._Max)
		aabb1._Min.X:-fTol
		aabb1._Min.Y:-fTol
		aabb1._Max.X:+fTol
		aabb1._Max.Y:+fTol
		aabb2._Min.X:-fTol
		aabb2._Min.Y:-fTol
		aabb2._Max.X:+fTol
		aabb2._Max.Y:+fTol
		If TAABB.Intersect(aabb1, aabb2) = False Then
			Return False
		End If
		Return True
	End Function
	
	rem
	Updates the values in the x and y extent lists by the changing aabb values.
	end rem
	Method UpdateExtentValues()
		Assert _xINfolist.Count() = _yInfoList.Count(), "x/y info list count does not match"
		For Local i:Int = 0 To _xInfoList.Count() - 1
			Local xInfo:TExtentInfo = TExtentInfo(_xInfoList.AtIndex(i))
			Local yInfo:TExtentInfo = TExtentInfo(_yInfoList.AtIndex(i))
			Assert xInfo._geometry = yInfo._geometry, "x/y info geometries do not match"
			Local aabb:TAABB = xInfo._geometry._aabb
			
			xInfo._Min._value = aabb._Min.X - fTol
			xInfo._max._value = aabb._Max.X + fTol
			yInfo._min._value = aabb._Min.Y - fTol
			yInfo._max._value = aabb._Max.Y + fTol
		Next
	End Method
	
	rem
	Iterates over the collision pairs and creates arbiters.
	end rem
	Method HandleCollisions()
		For Local cp:TCollisionPair = EachIn _collisionPairs.Keys()
			rem
			Note: Possible optimization. Maybe arbiter can be cached into value of
			collisionPairs? Currently, the collisionPairs hash doesn't use its
			value parameter - its just an unused bool value.
			end rem
			Local arbiter:TArbiter
			arbiter = TArbiter(_physics._arbiterPool.Fetch())
			arbiter.ConstructArbiter(cp.geom1, cp.geom2, _physics)
			If Not(_physics._arbiterList.Contains(arbiter)) Then
				_physics._arbiterList.Add(arbiter)
			Else
				_physics._arbiterPool.Free(arbiter)
			End If
		Next
	End Method
	
	Field _bForce:Int = False
	
	rem
	just calls update.
	end rem
	Method Run()
		If _bForce
			ForceNonIncrementalUpdate()
		Else
			Update()
		End If
	End Method
	
	rem
	Incrementall updates the system. assumes relatively good frame coherence.
	end rem
	Method Update()
		Self.UpdateExtentValues()
		_xExtentList.IncrementalSort()
		_yExtentList.IncrementalSort()

		_xInfoList.MoveUnderConsiderationToOverlaps()
		_yInfOlist.MoveUnderConsiderationToOverlaps()
		
		
		HandleCollisions()
	End Method
	
	Method ForceNonIncrementalUpdate()
		Self.UpdateExtentValues()
		' wipe out collision records
		_collisionPairs.Clear()
		Assert _xInfoList.Count() - _yInfoList.Count(), "x/y info list not same length"
		
		' clear out overlap records
		For Local ei:TExtentInfo = EachIn _xInfoList
			ei._overlaps.Clear()
			ei._underConsideration.Clear()
		Next
		For Local ei:TExtentInfo = EachIn _yInfoList
			ei._overlaps.Clear()
			ei._underConsideration.Clear()
		Next
		
		'force sort
		_xExtentList.Sort()
		_yExtentList.Sort()
		
		'rebuild overlap info
		Local overlaps:TList = CreateList()
		For Local i:Int = 0 To 1
			overlaps.Clear()
			
			Local extentList:TExtentList = Null
			If i = 0 Then
				extentList = _xExtentList
			Else
				extentList = _yExtentList
			End If
			
			For Local extent:TExtent = EachIn extentlist
				If extent._isMin Then
					'add whatever is currently in overlaps to this
					Local curNode:TLink = extent._info._overlaps.FirstLink()
					For Local ex:TExtent = EachIn overlaps
						extent._info._overlaps.InsertBeforeLink(ex, curNode)
					Next
					' now add, this geom to overlaps
					overlaps.AddLast(extent._info._geometry)
				Else
					' remove this geom from overlaps
					overlaps.Remove(extent._info._geometry)
					
					' test this geom against its overlaps for collisionpairs
					Local thisGeom:TGeom = extent._info._geometry
					For Local g:TGeom = EachIn extent._info._overlaps
						If DoCollision(thisGeom, g) = False Then
							Continue
						Else
							_collisionPairs.AddPair(thisGeom, g)
						EndIf
					Next
				End If
			Next
		Next
		HandleCollisions()
	End Method
End Type
